(function(utils, ImageViewController) {
    'use strict';

    /**
     * Image Controller useful for apply logic to all img elements
     * in the document.
     */
    function ImagesController() {
        /**
         * Keeps a reference of all the image view controllers
         * attached in the document.
         * @type {array.<ImageViewController>}
         */
        this.imgs = [];
    }

    /**
     * Find all img elements present in the document and attaches them
     * to an ImageViewController.
     */
    ImagesController.prototype.attachAllImages = function () {
        utils.onDomContentLoaded(function() {
            var imgNodeList = document.getElementsByTagName('img'),
                imgVC,
                i;

            for (i = 0; i < imgNodeList.length; i++) {
                imgVC = new ImageViewController();
                imgVC.attach(imgNodeList.item(i));

                this.imgs.push(imgVC);
            }
        }.bind(this));
    };

    /**
     * Refetch all the images from src.
     */
    ImagesController.prototype.retryAll = function() {
        utils.forEach(this.imgs, function(img) {
            img.retry();
        });
    };

    /**
     * Determines if each image in the document has a unique source or not.
     *
     * @returns {boolean} true if all images in the document have unique sources
     *                      else false.
     */
    ImagesController.prototype.isEveryImageSourceUnique = function() {
        var uniqueSources = Object.create(null),
            currentSource,
            i;

        for (i = 0; i < this.imgs.length; i++) {
            currentSource = this.imgs[i].el.getAttribute('src');
            if (currentSource && currentSource.length > 0 && uniqueSources[currentSource]) {
                // found a duplicate
                return false;
            }

            uniqueSources[currentSource] = true;
        }

        return true;
    };

    // export
    window.ImagesController = ImagesController;
})(window.utils, window.ImageViewController);
